#!/usr/bin/env python2
#
# Copyright (c) 2016,2018 Cisco and/or its affiliates.
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from string import Template

from jvpp_common_gen import generate_hash_code, generate_equals, generate_to_string, generate_fields
from jvpp_model import is_request, is_reply, is_retval, is_dump, is_details, is_event, is_control_ping, \
    is_control_ping_reply


def generate_dtos(work_dir, model, logger):
    logger.debug("Generating DTOs for %s " % model.json_api_files)
    _generate_message_dtos(work_dir, model, logger)
    _generate_dump_reply_wrappers(work_dir, model, logger)


def _generate_message_dtos(work_dir, model, logger):
    for msg in model.messages:
        logger.debug("Generating DTO for message %s", msg)
        class_name = msg.java_name_upper

        if is_control_ping(msg) or is_control_ping_reply(msg):
            # Skip control_ping managed by jvpp registry.
            continue
        if is_request(msg):
            dto = _generate_request_dto(msg, model, base_type="JVppRequest")
        elif is_dump(msg):
            dto = _generate_request_dto(msg, model, base_type="JVppDump")
        elif is_reply(msg) or is_details(msg):
            dto = _generate_reply_dto(msg, model)
        elif is_event(msg):
            dto = _generate_event_dto(msg, model)
        else:
            logger.warn("Failed to generate DTO for: %s. Message type is not supported." % msg)
            continue
        with open("%s/%s.java" % (work_dir, class_name), "w") as f:
            f.write(dto)


def _generate_request_dto(msg, model, base_type):
    msg_java_name_upper = msg.java_name_upper
    fields = msg.fields
    return _REQUEST_TEMPLATE.substitute(
        plugin_package=model.plugin_package,
        json_filename=model.json_api_files,
        json_definition=msg.doc,
        class_name=msg_java_name_upper,
        base_type=base_type,
        fields=generate_fields(fields),
        hash_code=generate_hash_code(fields),
        equals=generate_equals(msg_java_name_upper, fields),
        to_string=generate_to_string(msg_java_name_upper, fields),
        send=_generate_send(model, msg))

_REQUEST_TEMPLATE = Template("""
package $plugin_package.dto;

/**
 * <p>This class represents request DTO.
 * <br>It was generated by dto_gen.py based on $json_filename:
 * <pre>
$json_definition
 * </pre>
 */
public final class $class_name implements io.fd.vpp.jvpp.dto.$base_type {
$fields
$hash_code
$equals
$to_string
$send
}
""")


def _generate_send(model, msg):
    return _SEND_TEMPLATE.substitute(
        plugin_package=model.plugin_package,
        plugin_name=model.plugin_java_name,
        method_name=msg.java_name_lower,
        args="this" if msg.has_fields else ""
    )

_SEND_TEMPLATE = Template("""
    @Override
    public int send(final io.fd.vpp.jvpp.JVpp jvpp) throws io.fd.vpp.jvpp.VppInvocationException {
        return (($plugin_package.JVpp${plugin_name})jvpp).$method_name($args);
    }""")


def _generate_reply_dto(msg, model):
    msg_java_name_upper = msg.java_name_upper
    # Negative retval is mapped to java exception, so filter it out:
    fields = filter(lambda field: not is_retval(field), msg.fields)
    return _REPLY_TEMPLATE.substitute(
        plugin_package=model.plugin_package,
        json_filename=model.json_api_files,
        json_definition=msg.doc,
        class_name=msg_java_name_upper,
        request_name=msg.request_java,
        fields=generate_fields(fields),
        hash_code=generate_hash_code(fields),
        equals=generate_equals(msg_java_name_upper, fields),
        to_string=generate_to_string(msg_java_name_upper, fields))

_REPLY_TEMPLATE = Template("""
package $plugin_package.dto;

/**
 * <p>This class represents reply DTO.
 * <br>It was generated by jvpp_dto_gen.py based on $json_filename:
 * <pre>
$json_definition
 * </pre>
 */
public final class $class_name implements io.fd.vpp.jvpp.dto.JVppReply<$plugin_package.dto.$request_name> {
$fields
$hash_code
$equals
$to_string
}
""")


def _generate_event_dto(msg, model):
    msg_java_name_upper = msg.java_name_upper
    # Negative retval is mapped to java exception, so filter it out:
    fields = filter(lambda field: not is_retval(field), msg.fields)
    return _EVENT_TEMPLATE.substitute(
        plugin_package=model.plugin_package,
        json_filename=model.json_api_files,
        json_definition=msg.doc,
        class_name=msg_java_name_upper,
        fields=generate_fields(fields),
        hash_code=generate_hash_code(fields),
        equals=generate_equals(msg_java_name_upper, fields),
        to_string=generate_to_string(msg_java_name_upper, fields))

_EVENT_TEMPLATE = Template("""
package $plugin_package.dto;

/**
 * <p>This class represents event DTO.
 * <br>It was generated by jvpp_dto_gen.py based on $json_filename:
 * <pre>
$json_definition
 * </pre>
 */
public final class $class_name {
$fields
$hash_code
$equals
$to_string
}
""")


def _generate_dump_reply_wrappers(work_dir, model, logger):
    for msg in model.messages:
        if is_details(msg):
            logger.debug("Generating ReplyDump DTO for message %s", msg)
            details_class = msg.java_name_upper
            dto = _REPLY_DUMP_TEMPLATE.substitute(
                plugin_package=model.plugin_package,
                json_filename=model.json_api_files,
                json_definition=msg.doc,
                details_class=details_class,
                details_field=msg.java_name_lower,
                dump_class=msg.request_java
            )
            with open("%s/%sReplyDump.java" % (work_dir, details_class), "w") as f:
                f.write(dto)

_REPLY_DUMP_TEMPLATE = Template("""
package $plugin_package.dto;

/**
 * <p>This class represents dump reply wrapper.
 * <br>It was generated by jvpp_dto_gen.py based on $json_filename:
 * <pre>
$json_definition
 * </pre>
 */
public final class ${details_class}ReplyDump implements io.fd.vpp.jvpp.dto.JVppReplyDump<${plugin_package}.dto.${dump_class}, ${plugin_package}.dto.${details_class}> {

    public java.util.List<${details_class}> ${details_field} = new java.util.ArrayList<>();

    @Override
    @io.fd.vpp.jvpp.coverity.SuppressFBWarnings("UWF_UNWRITTEN_PUBLIC_OR_PROTECTED_FIELD")
    public int hashCode() {
        return java.util.Objects.hash(${details_field});
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        final ${details_class}ReplyDump other = (${details_class}ReplyDump) o;

        if (!java.util.Objects.equals(this.${details_field}, other.${details_field})) {
            return false;
        }

        return true;
    }

    @Override
    public java.lang.String toString() {
        return "${details_class}ReplyDump{" +
                "${details_field}=" + ${details_field} + "}";
    }


}
""")
