#!/usr/bin/env python2
#
# Copyright (c) 2018 Cisco and/or its affiliates.
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from string import Template

from jvpp_model import is_request, is_dump, is_event


def generate_java_ifc(work_dir, model, logger):
    logger.debug("Generating JVpp interface for %s" % model.json_api_files)
    messages = filter(_jvpp_ifc_filter, model.messages)
    plugin_package = model.plugin_package
    methods = []
    for msg in messages:
        if msg.has_fields:
            methods.append(_JVPP_IFC_METHOD_TEMPLATE.substitute(
                name=msg.java_name_lower,
                plugin_package=plugin_package,
                type=msg.java_name_upper))
        else:
            methods.append(_JVPP_IFC_NO_ARG_METHOD_TEMPLATE.substitute(name=msg.java_name_lower))

    plugin_name = model.plugin_java_name
    jvpp_interface = _JVPP_IFC_TEMPLATE.substitute(
        plugin_package=plugin_package,
        json_filename=model.json_api_files,
        plugin_name=plugin_name,
        methods="\n".join(methods)
    )
    with open("%s/JVpp%s.java" % (work_dir, plugin_name), "w") as f:
        f.write(jvpp_interface)


def _jvpp_ifc_filter(msg):
    return is_request(msg) or is_dump(msg) or is_event(msg)


_JVPP_IFC_METHOD_TEMPLATE = Template(
    """    int $name($plugin_package.dto.$type request) throws io.fd.vpp.jvpp.VppInvocationException;""")

_JVPP_IFC_NO_ARG_METHOD_TEMPLATE = Template("""    int $name() throws io.fd.vpp.jvpp.VppInvocationException;""")

_JVPP_IFC_TEMPLATE = Template("""package $plugin_package;

/**
 * <p>Java representation of plugin's api file.
 * <br>It was generated by jvpp_impl_gen.py based on $json_filename.
 * <br>(python representation of api file generated by vppapigen)
 */
public interface JVpp${plugin_name} extends io.fd.vpp.jvpp.JVpp {
    /**
     * Generic dispatch method for sending requests to VPP
     *
     * @throws io.fd.vpp.jvpp.VppInvocationException if send request had failed
     */
    int send(io.fd.vpp.jvpp.dto.JVppRequest request) throws io.fd.vpp.jvpp.VppInvocationException;
$methods
}
""")
