#!/usr/bin/env python2
#
# Copyright (c) 2016,2018 Cisco and/or its affiliates.
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
from string import Template

from jvpp_model import is_request, is_dump, is_event


def generate_java_impl(work_dir, model, logger):
    logger.debug("Generating JVpp implementation for %s" % model.json_api_files)
    messages = filter(_jvpp_impl_filter, model.messages)
    plugin_package = model.plugin_package
    methods = []
    for msg in messages:
        if msg.has_fields:
            methods.append(_JVPP_IMPL_METHOD_TEMPLATE.substitute(
                name=msg.java_name_lower,
                plugin_package=plugin_package,
                type=msg.java_name_upper))
        else:
            methods.append(_JVPP_IMPL_NO_ARG_METHOD_TEMPLATE.substitute(
                name=msg.java_name_lower,
                type=msg.java_name_upper))

    plugin_name = model.plugin_java_name
    jvpp_impl = _JVPP_IMPL_TEMPLATE.substitute(
        plugin_package=plugin_package,
        json_filename=model.json_api_files,
        plugin_name=model.plugin_java_name,
        plugin_name_underscore=model.plugin_name,
        methods="\n".join(methods))

    with open("%s/JVpp%sImpl.java" % (work_dir, plugin_name), "w") as f:
        f.write(jvpp_impl)


def _jvpp_impl_filter(msg):
    return is_request(msg) or is_dump(msg) or is_event(msg)


_JVPP_IMPL_TEMPLATE = Template("""package $plugin_package;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.Set;
import java.util.logging.Logger;
import java.util.logging.Level;
import io.fd.vpp.jvpp.callback.JVppCallback;
import io.fd.vpp.jvpp.VppConnection;
import io.fd.vpp.jvpp.JVppRegistry;

/**
 * <p>Default implementation of JVpp interface.
 * <br>It was generated by jvpp_impl_gen.py based on $json_filename.
 * <br>(python representation of api file generated by vppapigen)
 */
public final class JVpp${plugin_name}Impl implements $plugin_package.JVpp${plugin_name} {

    private final static Logger LOG = Logger.getLogger(JVpp${plugin_name}Impl.class.getName());
    private static final java.lang.String LIBNAME = "libjvpp_${plugin_name_underscore}.so";

    // FIXME using NativeLibraryLoader makes load fail could not find (WantInterfaceEventsReply).
    static {
        try {
            loadLibrary();
        } catch (Exception e) {
            LOG.severe("Can't find jvpp jni library: " + LIBNAME);
            throw new ExceptionInInitializerError(e);
        }
    }

    private static void loadStream(final InputStream is) throws IOException {
        final Set<PosixFilePermission> perms = PosixFilePermissions.fromString("rwxr-x---");
        final Path p = Files.createTempFile(LIBNAME, null, PosixFilePermissions.asFileAttribute(perms));
        try {
            Files.copy(is, p, StandardCopyOption.REPLACE_EXISTING);

            try {
                Runtime.getRuntime().load(p.toString());
            } catch (UnsatisfiedLinkError e) {
                throw new IOException("Failed to load library " + p, e);
            }
        } finally {
            try {
                Files.deleteIfExists(p);
            } catch (IOException e) {
            }
        }
    }

    private static void loadLibrary() throws IOException {
        try (final InputStream is = JVpp${plugin_name}Impl.class.getResourceAsStream('/' + LIBNAME)) {
            if (is == null) {
                throw new IOException("Failed to open library resource " + LIBNAME);
            }
            loadStream(is);
        }
    }

    private VppConnection connection;
    private JVppRegistry registry;

    private static native void init0(final JVppCallback callback, final long queueAddress, final int clientIndex);
    @Override
    public void init(final JVppRegistry registry, final JVppCallback callback, final long queueAddress, final int clientIndex) {
        this.registry = java.util.Objects.requireNonNull(registry, "registry should not be null");
        this.connection = java.util.Objects.requireNonNull(registry.getConnection(), "connection should not be null");
        connection.checkActive();
        init0(callback, queueAddress, clientIndex);
    }

    private static native void close0();
    @Override
    public void close() {
        close0();
    }

    @Override
    public int send(io.fd.vpp.jvpp.dto.JVppRequest request) throws io.fd.vpp.jvpp.VppInvocationException {
        return request.send(this);
    }

    @Override
    public final int controlPing(final io.fd.vpp.jvpp.dto.ControlPing controlPing) throws io.fd.vpp.jvpp.VppInvocationException {
        return registry.controlPing(JVpp${plugin_name}Impl.class);
    }
$methods
}
""")

_JVPP_IMPL_METHOD_TEMPLATE = Template("""
    private static native int ${name}0($plugin_package.dto.$type request);
    public final int $name($plugin_package.dto.$type request) throws io.fd.vpp.jvpp.VppInvocationException {
        java.util.Objects.requireNonNull(request, "Null request object");
        connection.checkActive();
        if (LOG.isLoggable(Level.FINE)) {
            LOG.fine(java.lang.String.format("Sending $type event message: %s", request));
        }
        int result=${name}0(request);
        if (result<0){
            throw new io.fd.vpp.jvpp.VppInvocationException("${name}", result);
        }
        return result;
    }""")

_JVPP_IMPL_NO_ARG_METHOD_TEMPLATE = Template("""
    private static native int ${name}0() throws io.fd.vpp.jvpp.VppInvocationException;
    public final int $name() throws io.fd.vpp.jvpp.VppInvocationException {
        connection.checkActive();
        LOG.fine("Sending $type event message");
        int result=${name}0();
        if(result<0){
            throw new io.fd.vpp.jvpp.VppInvocationException("${name}", result);
        }
        return result;
    }""")
