#!/usr/bin/env python2
#
# Copyright (c) 2018 Cisco and/or its affiliates.
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at:
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from string import Template

from jvpp_common_gen import generate_hash_code, generate_equals, generate_to_string, generate_fields
from jvpp_model import Union


def generate_unions(work_dir, model, logger):
    logger.debug("Generating unions for %s " % model.json_api_files)

    for t in model.types:
        if not isinstance(t, Union):
            continue
        logger.debug("Generating DTO for union %s", t)
        java_union_name = t.java_name
        fields = t.fields
        type_class = _UNION_TEMPLATE.substitute(
            plugin_package=model.plugin_package,
            c_type_name=t.name,
            json_filename=model.json_api_files,
            json_definition=t.doc,
            java_union_name=java_union_name,
            fields=generate_fields(fields, access_modifier="private"),
            constructors=_generate_constructors(java_union_name, fields),
            getters=_generate_getters(fields),
            hash_code=generate_hash_code(fields),
            equals=generate_equals(java_union_name, fields),
            to_string=generate_to_string(java_union_name, fields)
        )
        with open("%s/%s.java" % (work_dir, java_union_name), "w") as f:
            f.write(type_class)

_UNION_TEMPLATE = Template("""
package ${plugin_package}.types;

/**
 * <p>This class represents ${c_type_name} union definition.
 * <br>It was generated by unions_gen.py based on ${json_filename}:
 * <pre>
${json_definition}
 * </pre>
 */
public class ${java_union_name} {
    private final int _activeMember;
${fields}
    private ${java_union_name}() {
        // Constructor for JNI usage. All members can be read.
        _activeMember = -1;
    }
${constructors}
${getters}
${hash_code}
${equals}
${to_string}
}
""")


def _generate_constructors(union_name, fields):
    return "".join(
        _CONSTRUCTOR_TEMPLATE
        .substitute(union_name=union_name,
                    field_type=f.type.java_name_fqn,
                    field_name=f.java_name,
                    field_index=i) for i, f in enumerate(fields))

_CONSTRUCTOR_TEMPLATE = Template("""
    public ${union_name}(${field_type} ${field_name}) {
        this.${field_name} = java.util.Objects.requireNonNull(${field_name}, "${field_name} should not be null");
        _activeMember = $field_index;
    }""")


def _generate_getters(fields):
    return "".join(_GETTER_TEMPLATE.substitute(
        type=f.type.java_name_fqn,
        getter_name=f.java_name_upper,
        field_name=f.java_name
    ) for f in fields)

_GETTER_TEMPLATE = Template("""
    public ${type} get${getter_name}() {
        return ${field_name};
    }""")
