/*
 * Copyright (c) 2018 Cisco and/or its affiliates.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __VOM_IGMP_BINDING_H__
#define __VOM_IGMP_BINDING_H__

#include "vom/hw.hpp"
#include "vom/inspect.hpp"
#include "vom/interface.hpp"
#include "vom/object_base.hpp"
#include "vom/om.hpp"
#include "vom/singular_db.hpp"

namespace VOM {
/**
 * A representation of IGMP binding on an interface
 */
class igmp_binding : public object_base
{
public:
  /**
   * A binding is tied to a given interface, hence its key is
   * that of the interface
   */
  typedef interface::key_t key_t;

  /**
   * Construct a new object matching the desried state
   */
  igmp_binding(const interface& itf);

  /**
   * Copy Constructor
   */
  igmp_binding(const igmp_binding& o);

  /**
   * Destructor
   */
  ~igmp_binding();

  /**
   * Equal operator
   */
  bool operator==(const igmp_binding& l) const;

  /**
   * Get the object's key
   */
  const key_t key() const;

  /**
   * Return the 'singular' of the IGMP binding that matches this object
   */
  std::shared_ptr<igmp_binding> singular() const;

  /**
   * convert to string format for debug purposes
   */
  std::string to_string() const;

  /**
   * Return the matching'singular' of the interface
   */
  std::shared_ptr<interface> itf() const;

  /**
   * Dump all IGMP bindings into the stream provided
   */
  static void dump(std::ostream& os);

  /**
   * Find a listen from its key
   */
  static std::shared_ptr<igmp_binding> find(const key_t& k);

private:
  /**
   * Class definition for listeners to OM events
   */
  class event_handler : public OM::listener, public inspect::command_handler
  {
  public:
    event_handler();
    virtual ~event_handler() = default;

    /**
     * Handle a populate event
     */
    void handle_populate(const client_db::key_t& key);

    /**
     * Handle a replay event
     */
    void handle_replay();

    /**
     * Show the object in the Singular DB
     */
    void show(std::ostream& os);

    /**
     * Get the sortable Id of the listener
     */
    dependency_t order() const;
  };

  /**
   * event_handler to register with OM
   */
  static event_handler m_evh;

  /**
   * Enquue commonds to the VPP command Q for the update
   */
  void update(const igmp_binding& obj);

  /**
   * Find or add IGMP binding to the OM
   */
  static std::shared_ptr<igmp_binding> find_or_add(const igmp_binding& temp);

  /*
   * It's the OM class that calls singular()
   */
  friend class OM;

  /**
   * It's the singular_db class that calls replay()
   */
  friend class singular_db<key_t, igmp_binding>;

  /**
   * Sweep/reap the object if still stale
   */
  void sweep(void);

  /**
   * replay the object to create it in hardware
   */
  void replay(void);

  /**
   * A reference counting pointer to the interface on which IGMP config
   * resides. By holding the reference here, we can guarantee that
   * this object will outlive the interface
   */
  const std::shared_ptr<interface> m_itf;

  /**
   * HW configuration for the binding. The bool representing the
   * do/don't bind.
   */
  HW::item<bool> m_binding;

  /**
   * A map of all IGMP bindings keyed against the interface.
   */
  static singular_db<key_t, igmp_binding> m_db;
};
};

/*
 * fd.io coding-style-patch-verification: ON
 *
 * Local Variables:
 * eval: (c-set-style "mozilla")
 * End:
 */

#endif
