/*
 * decap.c : IPSec tunnel support
 *
 * Copyright (c) 2015 Cisco and/or its affiliates.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <vnet/vnet.h>
#include <vnet/api_errno.h>
#include <vnet/ip/ip.h>
#include <vnet/interface.h>

#include <vnet/ipsec/ipsec.h>

u8 *
format_ipsec_policy_action (u8 * s, va_list * args)
{
  u32 i = va_arg (*args, u32);
  char *t = 0;

  switch (i)
    {
#define _(v,f,str) case IPSEC_POLICY_ACTION_##f: t = str; break;
      foreach_ipsec_policy_action
#undef _
    default:
      s = format (s, "unknown");
    }
  s = format (s, "%s", t);
  return s;
}

uword
unformat_ipsec_policy_action (unformat_input_t * input, va_list * args)
{
  u32 *r = va_arg (*args, u32 *);

  if (0);
#define _(v,f,s) else if (unformat (input, s)) *r = IPSEC_POLICY_ACTION_##f;
  foreach_ipsec_policy_action
#undef _
    else
    return 0;
  return 1;
}

u8 *
format_ipsec_crypto_alg (u8 * s, va_list * args)
{
  u32 i = va_arg (*args, u32);
  u8 *t = 0;

  switch (i)
    {
#define _(v,f,str) case IPSEC_CRYPTO_ALG_##f: t = (u8 *) str; break;
      foreach_ipsec_crypto_alg
#undef _
    default:
      s = format (s, "unknown");
    }
  s = format (s, "%s", t);
  return s;
}

uword
unformat_ipsec_crypto_alg (unformat_input_t * input, va_list * args)
{
  u32 *r = va_arg (*args, u32 *);

  if (0);
#define _(v,f,s) else if (unformat (input, s)) *r = IPSEC_CRYPTO_ALG_##f;
  foreach_ipsec_crypto_alg
#undef _
    else
    return 0;
  return 1;
}

u8 *
format_ipsec_integ_alg (u8 * s, va_list * args)
{
  u32 i = va_arg (*args, u32);
  u8 *t = 0;

  switch (i)
    {
#define _(v,f,str) case IPSEC_INTEG_ALG_##f: t = (u8 *) str; break;
      foreach_ipsec_integ_alg
#undef _
    default:
      s = format (s, "unknown");
    }
  s = format (s, "%s", t);
  return s;
}

uword
unformat_ipsec_integ_alg (unformat_input_t * input, va_list * args)
{
  u32 *r = va_arg (*args, u32 *);

  if (0);
#define _(v,f,s) else if (unformat (input, s)) *r = IPSEC_INTEG_ALG_##f;
  foreach_ipsec_integ_alg
#undef _
    else
    return 0;
  return 1;
}

u8 *
format_ipsec_replay_window (u8 * s, va_list * args)
{
  u64 w = va_arg (*args, u64);
  u8 i;

  for (i = 0; i < 64; i++)
    {
      s = format (s, "%u", w & (1ULL << i) ? 1 : 0);
    }

  return s;
}

u8 *
format_ipsec_policy (u8 * s, va_list * args)
{
  u32 pi = va_arg (*args, u32);
  ipsec_main_t *im = &ipsec_main;
  ipsec_policy_t *p;
  vlib_counter_t counts;

  p = pool_elt_at_index (im->policies, pi);

  s = format (s, "  [%d] priority %d action %U protocol ",
	      pi, p->priority, format_ipsec_policy_action, p->policy);
  if (p->protocol)
    {
      s = format (s, "%U", format_ip_protocol, p->protocol);
    }
  else
    {
      s = format (s, "any");
    }
  if (p->policy == IPSEC_POLICY_ACTION_PROTECT)
    {
      s = format (s, " sa %u", p->sa_id);
    }
  if (p->is_ipv6)
    {
      s = format (s, "\n     local addr range %U - %U port range %u - %u",
		  format_ip6_address, &p->laddr.start.ip6,
		  format_ip6_address, &p->laddr.stop.ip6,
		  p->lport.start, p->lport.stop);
      s = format (s, "\n     remote addr range %U - %U port range %u - %u",
		  format_ip6_address, &p->raddr.start.ip6,
		  format_ip6_address, &p->raddr.stop.ip6,
		  p->rport.start, p->rport.stop);
    }
  else
    {
      s = format (s, "\n     local addr range %U - %U port range %u - %u",
		  format_ip4_address, &p->laddr.start.ip4,
		  format_ip4_address, &p->laddr.stop.ip4,
		  p->lport.start, p->lport.stop);
      s = format (s, "\n     remote addr range %U - %U port range %u - %u",
		  format_ip4_address, &p->raddr.start.ip4,
		  format_ip4_address, &p->raddr.stop.ip4,
		  p->rport.start, p->rport.stop);
    }
  vlib_get_combined_counter (&ipsec_spd_policy_counters, pi, &counts);
  s = format (s, "\n     packets %u bytes %u", counts.packets, counts.bytes);

  return (s);
}

u8 *
format_ipsec_spd (u8 * s, va_list * args)
{
  u32 si = va_arg (*args, u32);
  ipsec_main_t *im = &ipsec_main;
  ipsec_spd_t *spd;
  u32 *i;

  spd = pool_elt_at_index (im->spds, si);

  s = format (s, "spd %u", spd->id);

#define _(v, n)                                                 \
  s = format (s, "\n %s:", n);                                  \
  vec_foreach(i, spd->policies[IPSEC_SPD_POLICY_##v])           \
  {                                                             \
    s = format (s, "\n %U", format_ipsec_policy, *i);           \
  }
  foreach_ipsec_spd_policy_type;
#undef _

  return (s);
}

/*
 * fd.io coding-style-patch-verification: ON
 *
 * Local Variables:
 * eval: (c-set-style "gnu")
 * End:
 */
