/*
 *------------------------------------------------------------------
 * Copyright (c) 2017 Cisco and/or its affiliates.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *------------------------------------------------------------------
 */

#ifndef vpp_api_h_included
#define vpp_api_h_included

#include <string.h>
#include <stdbool.h>
#include <vppinfra/types.h>
#include <vlibapi/api_types.h>
#include <vapi/vapi_common.h>
#include <svm/queue.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @file vapi.h
 *
 * common vpp api C declarations
 *
 * This file declares the common C API functions. These include connect,
 * disconnect and utility functions as well as the low-level vapi_send and
 * vapi_recv API. This is only the transport layer.
 *
 * Message formats and higher-level APIs are generated by running the
 * vapi_c_gen.py script (which is run for in-tree APIs as part of the build
 * process). It's not recommended to mix the higher and lower level APIs. Due
 * to version issues, the higher-level APIs are not part of the shared library.
 */
  typedef struct vapi_ctx_s *vapi_ctx_t;

/**
 * @brief allocate vapi message of given size
 *
 * @note message must be freed by vapi_msg_free if not consumed by vapi_send
 * call
 *
 * @param ctx opaque vapi context
 *
 * @return pointer to message or NULL if out of memory
 */
  void *vapi_msg_alloc (vapi_ctx_t ctx, size_t size);

/**
 * @brief free a vapi message
 *
 * @note messages received by vapi_recv must be freed when no longer needed
 *
 * @param ctx opaque vapi context
 * @param msg message to be freed
 */
  void vapi_msg_free (vapi_ctx_t ctx, void *msg);

/**
 * @brief allocate vapi context
 *
 * @param[out] pointer to result variable
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_ctx_alloc (vapi_ctx_t * result);

/**
 * @brief free vapi context
 */
  void vapi_ctx_free (vapi_ctx_t ctx);

/**
 * @brief check if message identified by it's message id is known by the vpp to
 * which the connection is open
 */
  bool vapi_is_msg_available (vapi_ctx_t ctx, vapi_msg_id_t type);

/**
 * @brief connect to vpp
 *
 * @param ctx opaque vapi context, must be allocated using vapi_ctx_alloc first
 * @param name application name
 * @param chroot_prefix shared memory prefix
 * @param max_outstanding_requests max number of outstanding requests queued
 * @param response_queue_size size of the response queue
 * @param mode mode of operation - blocking or nonblocking
 * @param handle_keepalives - if true, automatically handle memclnt_keepalive
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_connect (vapi_ctx_t ctx, const char *name,
			     const char *chroot_prefix,
			     int max_outstanding_requests,
			     int response_queue_size, vapi_mode_e mode,
			     bool handle_keepalives);

/**
 * @brief disconnect from vpp
 *
 * @param ctx opaque vapi context
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_disconnect (vapi_ctx_t ctx);

/**
 * @brief get event file descriptor
 *
 * @note this file descriptor becomes readable when messages (from vpp)
 * are waiting in queue
 *
 * @param ctx opaque vapi context
 * @param[out] fd pointer to result variable
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_get_fd (vapi_ctx_t ctx, int *fd);

/**
 * @brief low-level api for sending messages to vpp
 *
 * @note it is not recommended to use this api directly, use generated api
 * instead
 *
 * @param ctx opaque vapi context
 * @param msg message to send
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_send (vapi_ctx_t ctx, void *msg);

/**
 * @brief low-level api for atomically sending two messages to vpp - either
 * both messages are sent or neither one is
 *
 * @note it is not recommended to use this api directly, use generated api
 * instead
 *
 * @param ctx opaque vapi context
 * @param msg1 first message to send
 * @param msg2 second message to send
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_send2 (vapi_ctx_t ctx, void *msg1, void *msg2);

/**
 * @brief low-level api for reading messages from vpp
 *
 * @note it is not recommended to use this api directly, use generated api
 * instead
 *
 * @param ctx opaque vapi context
 * @param[out] msg pointer to result variable containing message
 * @param[out] msg_size pointer to result variable containing message size
 * @param cond enum type for blocking, non-blocking or timed wait call
 * @param time in sec for timed wait
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_recv (vapi_ctx_t ctx, void **msg, size_t * msg_size,
			  svm_q_conditional_wait_t cond, u32 time);

/**
 * @brief wait for connection to become readable or writable
 *
 * @param ctx opaque vapi context
 * @param mode type of property to wait for - readability, writability or both
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_wait (vapi_ctx_t ctx, vapi_wait_mode_e mode);

/**
 * @brief pick next message sent by vpp and call the appropriate callback
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_dispatch_one (vapi_ctx_t ctx);

/**
 * @brief loop vapi_dispatch_one until responses to all currently outstanding
 * requests have been received and their callbacks called
 *
 * @note the dispatch loop is interrupted if any error is encountered or
 * returned from the callback, in which case this error is returned as the
 * result of vapi_dispatch. In this case it might be necessary to call dispatch
 * again to process the remaining messages. Returning VAPI_EUSER from
 * a callback allows the user to break the dispatch loop (and distinguish
 * this case in the calling code from other failures). VAPI never returns
 * VAPI_EUSER on its own.
 *
 * @return VAPI_OK on success, other error code on error
 */
  vapi_error_e vapi_dispatch (vapi_ctx_t ctx);

/** generic vapi event callback */
  typedef vapi_error_e (*vapi_event_cb) (vapi_ctx_t ctx, void *callback_ctx,
					 void *payload);

/**
 * @brief set event callback to call when message with given id is dispatched
 *
 * @param ctx opaque vapi context
 * @param id message id
 * @param callback callback
 * @param callback_ctx context pointer stored and passed to callback
 */
  void vapi_set_event_cb (vapi_ctx_t ctx, vapi_msg_id_t id,
			  vapi_event_cb callback, void *callback_ctx);

/**
 * @brief clear event callback for given message id
 *
 * @param ctx opaque vapi context
 * @param id message id
 */
  void vapi_clear_event_cb (vapi_ctx_t ctx, vapi_msg_id_t id);

/** generic vapi event callback */
  typedef vapi_error_e (*vapi_generic_event_cb) (vapi_ctx_t ctx,
						 void *callback_ctx,
						 vapi_msg_id_t id, void *msg);
/**
 * @brief set generic event callback
 *
 * @note this callback is called by dispatch if no message-type specific
 * callback is set (so it's a fallback callback)
 *
 * @param ctx opaque vapi context
 * @param callback callback
 * @param callback_ctx context pointer stored and passed to callback
 */
  void vapi_set_generic_event_cb (vapi_ctx_t ctx,
				  vapi_generic_event_cb callback,
				  void *callback_ctx);

/**
 * @brief clear generic event callback
 *
 * @param ctx opaque vapi context
 */
  void vapi_clear_generic_event_cb (vapi_ctx_t ctx);

#ifdef __cplusplus
}
#endif

#endif

/*
 * fd.io coding-style-patch-verification: ON
 *
 * Local Variables:
 * eval: (c-set-style "gnu")
 * End:
 */
